"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.logoutUser = exports.updateUserById = exports.getAllUsers = exports.deleteUserById = exports.getUserById = exports.loginUser = exports.registerUser = void 0;
const db_1 = __importDefault(require("../db/db"));
const schema_1 = require("../db/schema");
const drizzle_orm_1 = require("drizzle-orm");
const schema_2 = require("../validation/schema");
const authUtils_1 = require("../utils/authUtils");
const userCache = new Map();
// ✅ Register User
const registerUser = async (userData) => {
    schema_2.userSchema.parse(userData);
    const existingUser = await db_1.default
        .select()
        .from(schema_1.users)
        .where((0, drizzle_orm_1.or)((0, drizzle_orm_1.eq)(schema_1.users.email, userData.email), (0, drizzle_orm_1.eq)(schema_1.users.phone, userData.phone)))
        .limit(1);
    if (existingUser.length > 0) {
        if (existingUser[0].email === userData.email) {
            throw new Error("Email is already in use.");
        }
        if (existingUser[0].phone === userData.phone) {
            throw new Error("Phone number is already in use.");
        }
    }
    const hashedPassword = await (0, authUtils_1.hashPassword)(userData.password);
    const createdAt = new Date();
    try {
        const insertedIdResult = await db_1.default
            .insert(schema_1.users)
            .values({
            fullName: userData.fullName,
            email: userData.email,
            phone: userData.phone,
            passwordHash: hashedPassword,
            createdAt,
        })
            .$returningId();
        const insertedId = insertedIdResult[0]?.id;
        if (!insertedId) {
            throw new Error("Failed to retrieve inserted user ID.");
        }
        const [newUser] = await db_1.default
            .select()
            .from(schema_1.users)
            .where((0, drizzle_orm_1.eq)(schema_1.users.id, insertedId))
            .limit(1);
        if (!newUser) {
            throw new Error("Failed to retrieve newly created user.");
        }
        const safeUser = (0, authUtils_1.omitPassword)(newUser);
        const accessToken = (0, authUtils_1.generateToken)(safeUser.id, safeUser.email); // if generateToken expects both
        return { user: safeUser, accessToken };
    }
    catch (error) {
        if (error.message.includes("duplicate key value violates unique constraint") ||
            error.message.includes("Duplicate entry")) {
            if (error.message.includes("users_email_unique")) {
                throw new Error("Email is already in use.");
            }
            if (error.message.includes("users_phone_unique")) {
                throw new Error("Phone number is already in use.");
            }
        }
        throw error;
    }
};
exports.registerUser = registerUser;
// ✅ Login User
const loginUser = async (email, password) => {
    schema_2.loginSchema.parse({ email, password });
    const userData = await db_1.default
        .select({
        id: schema_1.users.id,
        email: schema_1.users.email,
        passwordHash: schema_1.users.passwordHash,
        fullName: schema_1.users.fullName,
    })
        .from(schema_1.users)
        .where((0, drizzle_orm_1.eq)(schema_1.users.email, email))
        .limit(1);
    if (userData.length === 0)
        throw new Error("User not found.");
    const userRecord = userData[0];
    userCache.set(email, userRecord);
    const isPasswordValid = await (0, authUtils_1.comparePassword)(password, userRecord.passwordHash);
    if (!isPasswordValid)
        throw new Error("Invalid credentials.");
    const safeUser = (0, authUtils_1.omitPassword)(userRecord);
    const accessToken = (0, authUtils_1.generateToken)(safeUser.id, safeUser.email);
    return { user: safeUser, accessToken };
};
exports.loginUser = loginUser;
// ✅ Get User by ID
const getUserById = async (requester, id) => {
    const userData = await db_1.default
        .select()
        .from(schema_1.users)
        .where((0, drizzle_orm_1.eq)(schema_1.users.id, id))
        .limit(1);
    if (userData.length === 0)
        throw new Error("User not found");
    return (0, authUtils_1.omitPassword)(userData[0]);
};
exports.getUserById = getUserById;
// ✅ Delete User by ID
const deleteUserById = async (id) => {
    await db_1.default.delete(schema_1.users).where((0, drizzle_orm_1.eq)(schema_1.users.id, id));
    return { deletedId: id };
};
exports.deleteUserById = deleteUserById;
// ✅ Get All Users
const getAllUsers = async (page = 1, limit = 10) => {
    const offset = (page - 1) * limit;
    const [userData, totalUsers] = await Promise.all([
        db_1.default
            .select()
            .from(schema_1.users)
            .limit(limit)
            .offset(offset)
            .orderBy((0, drizzle_orm_1.desc)(schema_1.users.createdAt)),
        db_1.default.select({ count: (0, drizzle_orm_1.count)() }).from(schema_1.users),
    ]);
    return {
        total: totalUsers[0].count,
        page,
        perPage: limit,
        users: userData.map(authUtils_1.omitPassword),
    };
};
exports.getAllUsers = getAllUsers;
// ✅ Update User
const updateUserById = async (userId, updateData) => {
    const filteredUpdateData = Object.fromEntries(Object.entries(updateData).filter(([_, value]) => value !== undefined));
    if (Object.keys(filteredUpdateData).length === 0) {
        throw new Error("No valid fields provided for update.");
    }
    await db_1.default
        .update(schema_1.users)
        .set(filteredUpdateData)
        .where((0, drizzle_orm_1.eq)(schema_1.users.id, userId));
    const [updatedUser] = await db_1.default
        .select()
        .from(schema_1.users)
        .where((0, drizzle_orm_1.eq)(schema_1.users.id, userId))
        .limit(1);
    if (!updatedUser)
        throw new Error("User not found or update failed.");
    return { updatedUser: (0, authUtils_1.omitPassword)(updatedUser) };
};
exports.updateUserById = updateUserById;
// ✅ Logout User
const logoutUser = async (email) => {
    userCache.delete(email);
};
exports.logoutUser = logoutUser;
