"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleUserRequest = exports.logoutController = exports.loginUserController = void 0;
const responseHelper_1 = require("../utils/responseHelper");
const authService_1 = require("../services/authService");
const modeValidator_1 = require("../utils/modeValidator");
// ✅ Login Controller (No token required)
const loginUserController = async (req, res) => {
    const { email, password } = req.body;
    if (!email || !password) {
        return (0, responseHelper_1.sendResponse)(res, 400, false, "Email and password are required.");
    }
    try {
        const loggedInUser = await (0, authService_1.loginUser)(email, password);
        return (0, responseHelper_1.sendResponse)(res, 200, true, "Login successful.", loggedInUser);
    }
    catch (error) {
        return (0, responseHelper_1.sendResponse)(res, 500, false, error.message || "Internal Server Error");
    }
};
exports.loginUserController = loginUserController;
// ✅ Logout Controller
const logoutController = async (req, res) => {
    const { email } = req.body;
    try {
        await (0, authService_1.logoutUser)(email);
        return (0, responseHelper_1.sendResponse)(res, 200, true, "Logout successful", null);
    }
    catch (error) {
        return (0, responseHelper_1.sendResponse)(res, 500, false, error.message || "Internal Server Error");
    }
};
exports.logoutController = logoutController;
// ✅ Handle Authenticated User Requests (Create, Retrieve, Update, Delete)
const handleUserRequest = async (req, res) => {
    const { mode, userId, page, limit, ...data } = req.body;
    const validModes = ["createnew", "update", "delete", "retrieve"];
    if (!(0, modeValidator_1.validateMode)(res, mode, validModes))
        return;
    try {
        switch (mode) {
            case "createnew": {
                const { fullName, email, password, phone } = data;
                if (!fullName || !email || !password || !phone) {
                    return (0, responseHelper_1.sendResponse)(res, 400, false, "Missing required fields: 'fullName', 'email', 'password', 'phone'.");
                }
                const newUser = await (0, authService_1.registerUser)({
                    fullName,
                    email,
                    password,
                    phone,
                });
                return (0, responseHelper_1.sendResponse)(res, 201, true, "User created successfully.", newUser);
            }
            case "retrieve": {
                if (userId) {
                    const user = await (0, authService_1.getUserById)(req.user, Number(userId));
                    return (0, responseHelper_1.sendResponse)(res, 200, true, "User retrieved successfully.", user);
                }
                else {
                    const pageNum = Number(page) || 1;
                    const limitNum = Number(limit) || 10;
                    if (pageNum < 1 || limitNum < 1) {
                        return (0, responseHelper_1.sendResponse)(res, 400, false, "Invalid pagination values.");
                    }
                    const users = await (0, authService_1.getAllUsers)(pageNum, limitNum);
                    return (0, responseHelper_1.sendResponse)(res, 200, true, "Users retrieved successfully.", users);
                }
            }
            case "update": {
                if (!userId) {
                    return (0, responseHelper_1.sendResponse)(res, 400, false, "User ID is required for updating.");
                }
                const updatedUser = await (0, authService_1.updateUserById)(Number(userId), data);
                return (0, responseHelper_1.sendResponse)(res, 200, true, "User updated successfully.", updatedUser);
            }
            case "delete": {
                if (!userId) {
                    return (0, responseHelper_1.sendResponse)(res, 400, false, "User ID is required for deletion.");
                }
                const result = await (0, authService_1.deleteUserById)(Number(userId));
                return (0, responseHelper_1.sendResponse)(res, 200, true, "User deleted successfully.", result);
            }
            default:
                return (0, responseHelper_1.sendResponse)(res, 400, false, "Unexpected error occurred.");
        }
    }
    catch (error) {
        return (0, responseHelper_1.sendResponse)(res, 500, false, error.message || "Internal Server Error");
    }
};
exports.handleUserRequest = handleUserRequest;
